//
//  Inwebo.swift
//  Demo iOS
//
//  Created by Benoit Vasseur on 28/06/2022.
//

import Foundation
import UIKit



class InweboService {
    
    private var iw: UnsafeMutablePointer<IW>! = nil
    
    private let DATA_FILE_NAME = "/DATA"
    
    // MARK: - Singleton
    
    public static let sharedInstance = InweboService()
    
    private let encoding = String.Encoding.isoLatin1
    
    private init() {
        
    }

    // MARK: - Init

    func initInwebo(macAccess: String, hostVersion: String) {
        let UID = strdup(UIDevice.current.identifierForVendor!.uuidString)
        defer { UID?.deallocate() }
        let SN = strdup("_")
        defer { SN?.deallocate() }
        let host = strdup("https://www.myinwebo.com")
        defer { host?.deallocate() }
        let hostVersionP = strdup(hostVersion)
        defer { hostVersionP?.deallocate() }
        let mac = strdup(macAccess)
        defer { mac?.deallocate() }
        let lang = strdup("fr")
        defer { lang?.deallocate() }
        
        
        self.iw = IWInit(0, UID, SN, {
            (url, timeout, user) in
            return WebserviceCall.webcall(urlString: url, timeout: timeout, user: user)
        }, nil)

        IWWsServerSet(iw, host)
        IWHostVersionSet(iw, hostVersionP)
        IWWsTimeoutSet(iw, 60000)
        IWMaccessSet(iw, mac)
        IWSetLang(iw, lang)

        // Creating the data file if it does not exist
        let path = NSSearchPathForDirectoriesInDomains(.documentDirectory, .userDomainMask, true)[0] as String
        if !FileManager.default.fileExists(atPath: path + DATA_FILE_NAME) {
            FileManager.default.createFile(atPath: path + DATA_FILE_NAME, contents: nil, attributes: [:])
        }
        
        // Init the session with what was store from the previous one
        let read = strdup(read())
        defer { read?.deallocate() }
        let storageDidSet = IWStorageDataSet(iw, read)
        print("did set storage : \(storageDidSet)")
    }
    
    // MARK: - Storage
    
    func updateStorage() {
        if IWStorageDataChanged(iw) > 0 {
            if let data = String(cString: IWStorageDataGet(iw), encoding: encoding) {
                write(data: data)
            }
            IWStorageDataRelease(iw)
        }
    }
    
    func read() -> String {
        let path = NSSearchPathForDirectoriesInDomains(.documentDirectory, .userDomainMask, true)[0] as String
        do {
            let content = try String(contentsOfFile: path + DATA_FILE_NAME)
            return content
        } catch {
            return ""
        }
    }
    
    func write(data: String) {
        let path = NSSearchPathForDirectoriesInDomains(.documentDirectory, .userDomainMask, true)[0] as String
        do {
            try data.write(toFile: path + DATA_FILE_NAME, atomically: true, encoding: .utf8)
        } catch {
            print("Error while writing");
        }
    }
    
    // MARK: - isActivated / isBlocked

    /**
     * Check if the app is already activated on this device
     * - Returns: True if the app if activated
     */
    func isActivated() -> Bool {
        return IWIsActivated(iw) != 0
    }
    
    /**
     * Check if the app is blocked and need to be unlocked
     * - Returns: True if the app if blocked
     */
    func isBlocked() -> Bool {
        return IWIsBlocked(iw) != 0
    }
    
    // MARK: - Activation
    
    /**
     * Start the activation process.
     * - Parameter code: the activation code
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func sendActivationCode(code: String) -> Int32 {
        let codeP = strdup(code)
        defer { codeP?.deallocate() }

        let result = IWActivationStart(iw, codeP)
        updateStorage()
        return result
    }
    
    /**
     * Finalize the activation process.
     * - Parameter code: the activation code
     * - Parameter pin: the user's pin code
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func finalizeActivationCode(code: String, pin: String) -> Int32 {
        let codeP = strdup(code)
        defer { codeP?.deallocate() }
        let pinP = strdup(pin)
        defer { pinP?.deallocate() }
        let nameP = strdup(UIDevice.current.name)
        defer { nameP?.deallocate() }
        
        let result = IWActivationFinalize(iw, codeP, pinP, nameP)
        updateStorage()
        return result
    }

    // MARK: - Pin Mode

    /**
     * Get the current Pin mode
     * - Returns: A value from the bitfield representing
     * the different Pin modes.
     * - IW_PINMODE_NONE = 0
     * - IW_PINMODE_CURRENT = 1
     * - IW_PINMODE_NEW = 2
     * - IW_PINMODE_BIO = 8
     */
    func
    getPinMode() -> Int32 {
        return IWPinMode(iw)
    }
    
    // MARK: - Synchronize

    /**
     * Start the synchronization process.
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func startSynchronize() -> Int32 {
        let result = IWSynchronizeStart(iw)
        updateStorage()
        return result
    }
    
    /**
     * Finalize the synchronization process.
     * - Parameter pin: the user's pin code
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func finalizeSynchronize(pin: String) -> Int32 {
        let pinP = strdup(pin)
        defer { pinP?.deallocate() }

        let result = IWSynchronizeFinalize(iw, pinP)
        updateStorage()
        return result
    }
    
    // MARK: - Pin change

    /**
     * Start the Pin modification process.
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func startChangePin() -> Int32 {
        let result = IWPwdUpdateStart(iw)
        updateStorage()
        return result
    }
    
    /**
     * Finalize the Pin modification process.
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func finalizeChangePin(currentPin: String, newPin: String) -> Int32 {
        let currentPinP = strdup(currentPin)
        defer { currentPinP?.deallocate() }
        let newPinP = strdup(newPin)
        defer { newPinP?.deallocate() }

        let result = IWPwdUpdateFinalize(iw, newPinP, currentPinP)
        updateStorage()
        return result
    }
    
    // MARK: - OTP
    
    /**
     * Start the online OTP generation process.
     * - Warning: Should only be used during online workflow
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func startOnlineOTPGenerationProcess() -> Int32 {
        let result = IWOnlineOtpStart(iw, 0)
        updateStorage()
        return result
    }
    
    /**
     * Finalize the online OTP generation process.
     * If the result is IW_ERR_OK, the OTP can be retrieved using
     * `getOtpAnswer()`
     * - Warning: Should only be used during online workflow
     * - Parameter pin: the user's pin code
     * - Parameter isBiokey: if the pin used corresponds to the biokey
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func finalizeOnlineOTPGenerationProcess(pin: String, isBiokey: Bool = false) -> Int32 {
        let pinP = strdup(pin)
        defer { pinP?.deallocate() }

        let result = isBiokey ? IWOnlineOtpFinalizeExt(iw, 0, pinP, 1) : IWOnlineOtpFinalize(iw, 0, pinP)
        updateStorage()
        return result
    }
    
    /**
     * Retrieve the current OTP generated by the last successful
     * call to `finalizeOTPGenerationProcess()`
     * - Warning: Should only be used during online workflow
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func getOnlineOtpAnswer() -> String? {
        guard let otp = IWOtpAnswerOtp(iw) else {
            return nil
        }
        return String(cString: otp, encoding: encoding)
    }
    
    /**
     * Check if the app is still synchronized and can be
     * used for offline otp generation
     * - Returns: true if synchronized
     */
    func otpShouldSynchronize() -> Bool {
        return IWOtpShouldSynchronize(iw, 0) == 1
    }
    
    /**
     * Use to check if the pin code is required
     * - Warning: Should only be used during offline workflow
     * - Returns: true if pin code is required
     */
    func isOfflineOtpPinRequired() -> Bool {
        return IWOtpModeQuery(iw, 0) == 1
    }
    
    /**
     * Generate an offline OTP
     * - Warning: Should only be used during offline workflow
     * - Returns: The OTP generated
     */
    func getOfflineOtpGenerate(pin: String) -> String? {
        let pinP = strdup(pin)
        defer { pinP?.deallocate() }

        guard let otp = IWOtpGenerateMa(iw, pinP) else {
            return nil
        }
        return String(cString: otp, encoding: encoding)
    }
    
    // MARK: - Seal
    
    /**
     * Start the online Seal generation process.
     * - Warning: Should only be used during online workflow
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func startOnlineSealGenerationProcess() -> Int32 {
        let result = IWOnlineSealStart(iw, 0)
        updateStorage()
        return result
    }
    
    /**
     * Finalize the online Seal generation process.
     * If the result is IW_ERR_OK, the Seal can be retrieved using
     * `getSealAnswser()`
     * - Warning: Should only be used during online workflow
     * - Parameter pin: the user's pin code
     * - Parameter toSeal: the data to seal
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func finalizeOnlineSealGenerationProcess(pin: String, toSeal: String) -> Int32 {
        let pinP = strdup(pin)
        defer { pinP?.deallocate() }
        let toSealP = strdup(toSeal)
        defer { toSealP?.deallocate() }

        let result = IWOnlineSealFinalize(iw, 0, pinP, toSealP)
        updateStorage()
        return result
    }
    
    /**
     * Retrieve the current Seal generated by the last successful
     * call to `finalizeSealGenerationProcess()`
     * - Warning: Should only be used during online workflow
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func getOnlineSealAnswser() -> String? {
        guard let seal = IWSealAnswerOtp(iw) else {
            return nil
        }
        return String(cString: seal, encoding: encoding)
    }
    
    /**
     * Check if the app is still synchronized and can be
     * used for offline seal generation
     * - Returns: true if synchronized
     */
    func sealShouldSynchronize() -> Bool {
        return IWSealShouldSynchronize(iw, 0) == 1
    }
    
    /**
     * Use to check if the pin code is required
     * - Warning: Should only be used during offline workflow
     * - Returns: true if pin code is required
     */
    func isOfflineSealPinRequired() -> Bool {
        return IWSealModeQuery(iw, 0) == 1
    }
    
    /**
     * Generate an offline OTP
     * - Warning: Should only be used during offline workflow
     * - Returns: The OTP generated
     */
    func getOfflineSealGenerate(pin: String, toSeal: String) -> String? {
        let pinP = strdup(pin)
        defer { pinP?.deallocate() }
        let toSealP = strdup(toSeal)
        defer { toSealP?.deallocate() }

        guard let seal = IWSealGenerate(iw, pinP, toSealP) else {
            return nil
        }
        return String(cString: seal, encoding: encoding)
    }
    
    // MARK: - Unlock
    
    /**
     * Start the unlock app process.
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func startUnlock(code: String) -> Int32 {
        let codeP = strdup(code)
        defer { codeP?.deallocate() }
        
        let result = IWResetStart(iw, codeP)
        updateStorage()
        return result
    }
    
    /**
     * Finalize the unlock process.
     * - Parameter code: the unlock code
     * - Parameter pin: the user's pin code
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func finalizeUnlock(code: String, pin: String) -> Int32 {
        let pinP = strdup(pin)
        defer { pinP?.deallocate() }
        let codeP = strdup(code)
        defer { codeP?.deallocate() }
        
        let result = IWResetFinalize(iw, codeP, pinP)
        updateStorage()
        return result
    }
    
    // MARK: - Biokey
    
    /**
     * Start the register biokey process.
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func startRegisterBiokey() -> Int32 {
        let result = IWSetBiokeyStart(iw)
        updateStorage()
        return result
    }
    
    /**
     * Finalize the register biokey process.
     * - Parameter biokey: the biokey
     * - Parameter pin: the user's pin code
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func finalizeRegisterBiokey(biokey: String, pin: String) -> Int32 {
        let pinP = strdup(pin)
        defer { pinP?.deallocate() }
        let biokeyP = strdup(biokey)
        defer { biokeyP?.deallocate() }
        
        let result = IWSetBiokeyFinalize(iw, biokeyP, pinP)
        updateStorage()
        return result
    }
    
    /**
     * Start the unset biokey process.
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func startUnsetBiokey() -> Int32 {
        let result = IWUnsetBiokeysStart(iw)
        updateStorage()
        return result
    }
    
    /**
     * Finalize the unset biokey process.
     * - Parameter pin: the user's pin code
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func finalizeUnsetBiokey(pin: String) -> Int32 {
        let pinP = strdup(pin)
        defer { pinP?.deallocate() }

        let result = IWUnsetBiokeysFinalize(iw, pinP)
        updateStorage()
        return result
    }
    
    // MARK: - Notifications
    
    /**
     * Indicate the push system to use
     * - Parameter os: the os of the device
     * or firebase for a generic config
     */
    func setDeviceOS(os: String) {
        let osP = strdup(os)
        defer { osP?.deallocate() }
        
        IWSetDeviceOS(iw, osP)
    }
    
    /**
     * Start the push registration process
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func startPushRegistration() -> Int32 {
        let result = IWPushRegistrationStart(iw)
        updateStorage()
        return result
    }
    
    /**
     * Finalize the push registration process
     * - Parameter token: the FCM (firebase cloud messaging) token
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func finalizePushRegistration(token: String) -> Int32 {
        let tokenP = strdup(token)
        defer { tokenP?.deallocate() }
        
        let result = IWPushRegistrationFinalize(iw, tokenP)
        updateStorage()
        return result
    }
    
    /**
     * Fetch pending push
     * Push info can then be retrieved with
     * getPushAlias, getPushContext and getPushAction
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func checkPush() -> Int32 {
        let result = IWCheckPush(iw)
        updateStorage()
        return result
    }
    
    /**
     * Get the pending push alias
     * - Warning: A call to checkPush is necessary before calling this action
     * - Returns: the pending push alias
     */
    func getPushAlias() -> String? {
        if let alias = IWPushAlias(iw) {
            return String(cString: alias, encoding: encoding)
        }
        return nil
    }
    
    /**
     * Get the pending push contect
     * - Warning: A call to checkPush is necessary before calling this action
     * - Returns: the pending push context
     */
    func getPushContext() -> String? {
        if let context = IWPushContext(iw) {
            return String(cString: context, encoding: encoding)
        }
        return nil
    }
    
    /**
     * Get the pending push action
     * - Warning: A call to checkPush is necessary before calling this action
     * - Returns: the pending push action
     */
    func getPushAction() -> String? {
        if let action = IWPushAction(iw) {
            return String(cString: action, encoding: encoding)
        }
        return nil
    }
    
    /**
     * Start the Push OTP process
     * - Parameter alias: the alias returned by a call to getPushAlias
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func startPushOTP(alias: String) -> Int32 {
        let aliasP = strdup(alias)
        defer { aliasP?.deallocate() }
        
        let result = IWPushOtpStart(iw, aliasP)
        updateStorage()
        return result
    }
    
    /**
     * Finalize the Push OTP process
     * - Parameter alias: the alias returned by a call to getPushAlias
     * - Parameter pin: the user's pin code
     * - Parameter ok: if the user wishes to validate the push OTP process
     * - Parameter isBio: if the pin field is a biokey
     * - Returns: IW_ERR_OK if ok, an error code otherwise
     */
    func finalizePushOTP(alias: String, pin: String, ok: Bool, isBio: Bool) -> Int32 {
        let aliasP = strdup(alias)
        defer { aliasP?.deallocate() }
        let pinP = strdup(pin)
        defer { pinP?.deallocate() }
        let result = isBio ? IWPushOtpFinalizeExt(iw, aliasP, pinP, ok ? 1 : 0, 1) : IWPushOtpFinalize(iw, aliasP, pinP, ok ? 1 : 0)
        updateStorage()
        return result
    }
    
    // MARK: - Utils
    
    /**
     * Generate a countdown timestamp
     * - Returns: A countdown timestamp in seconds
     */
    func countdown() -> TimeInterval {
        return TimeInterval(IWDisplayTime(iw))
    }
    
    // MARK: - Error

    func errorMessageFor(code: Int32) -> String {
        switch code {
            
        case IW_ERR_NETWORK:
            return "Network unreachable or incorrect phone configuration"
        case IW_ERR_CODE:
            return "This code has not been recognized or is no longer valid"
        case IW_ERR_SN:
            return "Incorrect value"
        case IW_ERR_ACCESS:
            return "The PIN entered is incorrect"
        case IW_ERR_VERSION:
            return "this authenticator version is obsolete"
        case IW_ERR_BLOCKED:
            return "You user account is locked"
        case IW_ERR_NODEVICE:
            return "Application has been deactivated or suppressed"
        case IW_ERR_MAXNBTOOLS:
            return "Activation failed; the maximum number of allowed mobile apps has been reached"
        case IW_ERR_OTHER:
            return "The operation has failed"
        case IW_ERR_FORBIDDEN:
            return "This action is forbidden"
        case IW_ERR_BIOKEY:
            return "Initialize your biokey first"
        default:
            return "The operation has failed"
        }
    }

}
